@extends('layouts.admin')

@section('title', 'Plugin Details - ' . ($pluginData['display_name'] ?? $pluginData['name']))

@section('content')
<div class="dashboard-header">
    <div class="d-flex justify-content-between align-items-center">
        <div>
            <h1>
                <i class="fas fa-puzzle-piece"></i> 
                {{ $pluginData['display_name'] ?? $pluginData['name'] }}
            </h1>
            <p>{{ $pluginData['description'] ?? 'No description available' }}</p>
        </div>
        <div class="plugin-actions">
            @if($pluginData['status'] === 'active')
                <form action="{{ route('admin.plugins.deactivate', $pluginData['name']) }}" method="POST" class="d-inline">
                    @csrf
                    <button type="submit" class="btn btn-warning">
                        <i class="fas fa-pause"></i> Deactivate
                    </button>
                </form>
            @else
                <form action="{{ route('admin.plugins.activate', $pluginData['name']) }}" method="POST" class="d-inline">
                    @csrf
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-play"></i> Activate
                    </button>
                </form>
            @endif
            
            <a href="{{ route('admin.plugins.index') }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Back to Plugins
            </a>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-md-8">
        <!-- Plugin Configuration -->
        @if($pluginData['status'] === 'active' && isset($pluginData['config_schema']) && !empty($pluginData['config_schema']))
        <div class="dashboard-card">
            <div class="dashboard-card-header">
                <h3><i class="fas fa-cogs"></i> Configuration</h3>
            </div>
            <div class="dashboard-card-body">
                <form action="{{ route('admin.plugins.config.update', $pluginData['name']) }}" method="POST" id="configForm">
                    @csrf
                    <div class="config-grid">
                        @foreach($pluginData['config_schema'] as $key => $schema)
                        <div class="config-item">
                            <div class="config-label">
                                <label for="config_{{ $key }}">{{ $schema['label'] ?? ucfirst(str_replace('_', ' ', $key)) }}</label>
                                @if(isset($schema['description']))
                                    <small class="text-muted">{{ $schema['description'] }}</small>
                                @endif
                                <span class="config-key">{{ $key }}</span>
                            </div>
                            <div class="config-input">
                                <span class="config-type">{{ $schema['type'] ?? 'text' }}</span>
                                @switch($schema['type'] ?? 'text')
                                    @case('boolean')
                                        <div class="toggle-switch">
                                            <input type="checkbox" 
                                                   class="toggle-switch-checkbox" 
                                                   id="config_{{ $key }}" 
                                                   name="config[{{ $key }}]" 
                                                   value="1"
                                                   {{ ($pluginData['config'][$key] ?? $schema['default'] ?? false) ? 'checked' : '' }}>
                                            <label for="config_{{ $key }}" class="toggle-switch-label">
                                                <span class="toggle-switch-switch"></span>
                                            </label>
                                        </div>
                                        @break
                                    
                                    @case('select')
                                        <select name="config[{{ $key }}]" id="config_{{ $key }}" class="form-control">
                                            @foreach($schema['options'] ?? [] as $optionValue => $optionLabel)
                                                <option value="{{ $optionValue }}" 
                                                    {{ ($pluginData['config'][$key] ?? $schema['default'] ?? '') == $optionValue ? 'selected' : '' }}>
                                                    {{ $optionLabel }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @break
                                    
                                    @case('textarea')
                                        <textarea name="config[{{ $key }}]" 
                                                  id="config_{{ $key }}" 
                                                  class="form-control" 
                                                  rows="4"
                                                  placeholder="{{ $schema['placeholder'] ?? '' }}">{{ $pluginData['config'][$key] ?? $schema['default'] ?? '' }}</textarea>
                                        @break
                                    
                                    @case('json')
                                        <div class="json-editor">
                                            <textarea name="config[{{ $key }}]" 
                                                      id="config_{{ $key }}" 
                                                      class="form-control json-textarea" 
                                                      rows="6"
                                                      placeholder="{{ $schema['placeholder'] ?? 'Enter JSON configuration...' }}">{{ is_array($pluginData['config'][$key] ?? $schema['default'] ?? []) ? json_encode($pluginData['config'][$key] ?? $schema['default'] ?? [], JSON_PRETTY_PRINT) : ($pluginData['config'][$key] ?? $schema['default'] ?? '') }}</textarea>
                                            <button type="button" class="format-json-btn" data-target="config_{{ $key }}">
                                                <i class="fas fa-code"></i> Format JSON
                                            </button>
                                        </div>
                                        @break
                                    
                                    @case('color')
                                        <div class="color-picker-wrapper">
                                            <input type="color" 
                                                   class="color-picker-input" 
                                                   id="color_{{ $key }}" 
                                                   value="{{ $pluginData['config'][$key] ?? $schema['default'] ?? '#000000' }}">
                                            <input type="text" 
                                                   name="config[{{ $key }}]" 
                                                   class="form-control color-text-input" 
                                                   value="{{ $pluginData['config'][$key] ?? $schema['default'] ?? '#000000' }}"
                                                   placeholder="#000000">
                                        </div>
                                        @break
                                    
                                    @case('number')
                                        <input type="number" 
                                               name="config[{{ $key }}]" 
                                               id="config_{{ $key }}" 
                                               class="form-control" 
                                               value="{{ $pluginData['config'][$key] ?? $schema['default'] ?? '' }}"
                                               min="{{ $schema['min'] ?? '' }}"
                                               max="{{ $schema['max'] ?? '' }}"
                                               step="{{ $schema['step'] ?? '1' }}"
                                               placeholder="{{ $schema['placeholder'] ?? '' }}">
                                        @break
                                    
                                    @default
                                        <input type="text" 
                                               name="config[{{ $key }}]" 
                                               id="config_{{ $key }}" 
                                               class="form-control" 
                                               value="{{ $pluginData['config'][$key] ?? $schema['default'] ?? '' }}"
                                               placeholder="{{ $schema['placeholder'] ?? '' }}">
                                @endswitch
                            </div>
                        </div>
                        @endforeach
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Save Configuration
                        </button>
                        <button type="button" class="btn btn-secondary" id="resetConfigBtn">
                            <i class="fas fa-undo"></i> Reset to Defaults
                        </button>
                    </div>
                </form>
            </div>
        </div>
        @elseif($pluginData['status'] === 'active')
        <div class="dashboard-card">
            <div class="dashboard-card-header">
                <h3><i class="fas fa-cogs"></i> Configuration</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="text-center py-4">
                    <i class="fas fa-cog fa-3x text-muted mb-3"></i>
                    <h5>No Configuration Available</h5>
                    <p class="text-muted">This plugin doesn't have any configurable settings.</p>
                </div>
            </div>
        </div>
        @endif
        
        <!-- Plugin Information -->
        <div class="dashboard-card">
            <div class="dashboard-card-header">
                <h3><i class="fas fa-info-circle"></i> Plugin Information</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="plugin-info-grid">
                    <div class="info-item">
                        <label>Name:</label>
                        <span>{{ $pluginData['display_name'] ?? $pluginData['name'] }}</span>
                    </div>
                    <div class="info-item">
                        <label>Version:</label>
                        <span>{{ $pluginData['version'] ?? '1.0.0' }}</span>
                    </div>
                    <div class="info-item">
                        <label>Author:</label>
                        <span>{{ $pluginData['author'] ?? 'Unknown' }}</span>
                    </div>
                    <div class="info-item">
                        <label>Status:</label>
                        <span class="status-badge status-{{ $pluginData['status'] }}">
                            <i class="fas fa-{{ $pluginData['status'] === 'active' ? 'check-circle' : 'pause-circle' }}"></i>
                            {{ ucfirst($pluginData['status']) }}
                        </span>
                    </div>
                    @if(isset($pluginData['website']))
                    <div class="info-item">
                        <label>Website:</label>
                        <a href="{{ $pluginData['website'] }}" target="_blank" class="plugin-link">
                            {{ $pluginData['website'] }} <i class="fas fa-external-link-alt"></i>
                        </a>
                    </div>
                    @endif
                    @if(isset($pluginData['license']))
                    <div class="info-item">
                        <label>License:</label>
                        <span>{{ $pluginData['license'] }}</span>
                    </div>
                    @endif
                    <div class="info-item">
                        <label>Installation Date:</label>
                        <span>{{ $pluginData['installed_at'] ?? 'Unknown' }}</span>
                    </div>
                    @if(isset($pluginData['last_updated']))
                    <div class="info-item">
                        <label>Last Updated:</label>
                        <span>{{ $pluginData['last_updated'] }}</span>
                    </div>
                    @endif
                </div>
                
                @if(isset($pluginData['requirements']) && !empty($pluginData['requirements']))
                <div class="mt-4">
                    <h5>Requirements:</h5>
                    <div class="requirements-list">
                        @foreach($pluginData['requirements'] as $requirement => $version)
                        <div class="requirement-item">
                            <i class="fas fa-check-circle text-success"></i>
                            <span>{{ $requirement }}: {{ $version }}</span>
                        </div>
                        @endforeach
                    </div>
                </div>
                @endif
            </div>
        </div>
        
        <!-- Plugin Files -->
        @if(isset($pluginData['files']) && count($pluginData['files']) > 0)
        <div class="dashboard-card">
            <div class="dashboard-card-header">
                <h3><i class="fas fa-folder"></i> Plugin Files</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="plugin-files-tree">
                    @foreach($pluginData['files'] as $file)
                    <div class="file-item">
                        <i class="fas fa-{{ str_contains($file, '.php') ? 'file-code' : (str_contains($file, '.js') ? 'file-code' : (str_contains($file, '.css') ? 'file-code' : 'file')) }}"></i>
                        <span>{{ $file }}</span>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
        @endif
    </div>
    
    <div class="col-md-4">
        <!-- Quick Actions -->
        <div class="dashboard-card">
            <div class="dashboard-card-header">
                <h3><i class="fas fa-bolt"></i> Quick Actions</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="quick-actions">
                    @if($pluginData['has_update'] ?? false)
                    <button class="quick-action-btn update-plugin" data-plugin="{{ $pluginData['name'] }}">
                        <i class="fas fa-download"></i>
                        <span>Update Plugin</span>
                    </button>
                    @endif
                    
                    <a href="{{ route('admin.plugins.config.export', $pluginData['name']) }}" class="quick-action-btn">
                        <i class="fas fa-download"></i>
                        <span>Export Config</span>
                    </a>
                    
                    <button class="quick-action-btn" id="importConfigBtn">
                        <i class="fas fa-upload"></i>
                        <span>Import Config</span>
                    </button>
                    
                    <button class="quick-action-btn text-danger" id="uninstallPluginBtn">
                        <i class="fas fa-trash"></i>
                        <span>Uninstall Plugin</span>
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Plugin Statistics -->
        @if(isset($pluginData['stats']) && !empty($pluginData['stats']))
        <div class="dashboard-card">
            <div class="dashboard-card-header">
                <h3><i class="fas fa-chart-bar"></i> Statistics</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="plugin-stats">
                    @foreach($pluginData['stats'] as $stat => $value)
                    <div class="stat-row">
                        <span class="stat-label">{{ ucfirst(str_replace('_', ' ', $stat)) }}:</span>
                        <span class="stat-value">{{ $value }}</span>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
        @endif
        
        <!-- Plugin Logs -->
        @if(isset($pluginData['logs']) && count($pluginData['logs']) > 0)
        <div class="dashboard-card">
            <div class="dashboard-card-header">
                <h3><i class="fas fa-list-alt"></i> Recent Logs</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="plugin-logs">
                    @foreach(array_slice($pluginData['logs'], 0, 5) as $log)
                    <div class="log-item">
                        <div class="log-level log-level-{{ $log['level'] ?? 'info' }}">
                            {{ strtoupper($log['level'] ?? 'info') }}
                        </div>
                        <div class="log-content">
                            <div class="log-message">{{ $log['message'] }}</div>
                            <div class="log-time">{{ $log['timestamp'] ?? 'Unknown' }}</div>
                        </div>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
        @endif
    </div>
</div>

<!-- Import Config Modal -->
<div class="modal-backdrop" id="importConfigModal">
    <div class="modal">
        <div class="modal-header">
            <h3><i class="fas fa-upload"></i> Import Configuration</h3>
            <button class="close-btn" type="button">&times;</button>
        </div>
        <div class="modal-body">
            <form id="importConfigForm" enctype="multipart/form-data">
                @csrf
                <div class="form-group">
                    <label for="configFile">Configuration File (JSON)</label>
                    <div class="file-upload-wrapper">
                        <input type="file" id="configFile" name="config_file" accept=".json" class="file-upload-input">
                        <div class="file-upload-label">
                            <i class="fas fa-upload file-upload-icon"></i>
                            <span>Choose JSON configuration file</span>
                        </div>
                    </div>
                </div>
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    Importing a configuration will overwrite current settings. Make sure to export your current configuration first if needed.
                </div>
            </form>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary close-modal">Cancel</button>
            <button class="btn btn-primary" id="confirmImportBtn">Import Configuration</button>
        </div>
    </div>
</div>

<!-- Uninstall Confirmation Modal -->
<div class="modal-backdrop" id="uninstallModal">
    <div class="modal">
        <div class="modal-header">
            <h3><i class="fas fa-exclamation-triangle text-danger"></i> Confirm Uninstall</h3>
            <button class="close-btn" type="button">&times;</button>
        </div>
        <div class="modal-body">
            <p>Are you sure you want to uninstall the plugin "<strong>{{ $pluginData['display_name'] ?? $pluginData['name'] }}</strong>"?</p>
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-triangle"></i>
                <strong>Warning:</strong> This action cannot be undone. All plugin data, configuration, and files will be permanently removed.
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary close-modal">Cancel</button>
            <button class="btn btn-danger" id="confirmUninstallBtn">Uninstall Plugin</button>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Color picker synchronization
    document.querySelectorAll('.color-picker-wrapper').forEach(wrapper => {
        const colorInput = wrapper.querySelector('.color-picker-input');
        const textInput = wrapper.querySelector('.color-text-input');
        
        if (colorInput && textInput) {
            colorInput.addEventListener('input', function() {
                textInput.value = this.value;
            });
            
            textInput.addEventListener('input', function() {
                if (/^#[0-9A-F]{6}$/i.test(this.value)) {
                    colorInput.value = this.value;
                }
            });
        }
    });
    
    // JSON formatting
    document.querySelectorAll('.format-json-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const targetId = this.dataset.target;
            const textarea = document.getElementById(targetId);
            
            try {
                const parsed = JSON.parse(textarea.value);
                textarea.value = JSON.stringify(parsed, null, 2);
                showAlert('success', 'JSON formatted successfully!');
            } catch (e) {
                showAlert('error', 'Invalid JSON format!');
            }
        });
    });
    
    // Reset configuration to defaults
    document.getElementById('resetConfigBtn')?.addEventListener('click', function() {
        if (confirm('Are you sure you want to reset all configuration to default values?')) {
            const form = document.getElementById('configForm');
            const inputs = form.querySelectorAll('input, select, textarea');
            
            inputs.forEach(input => {
                const defaultValue = input.getAttribute('data-default') || '';
                
                if (input.type === 'checkbox') {
                    input.checked = defaultValue === 'true' || defaultValue === '1';
                } else {
                    input.value = defaultValue;
                }
            });
            
            showAlert('info', 'Configuration reset to default values. Don\'t forget to save!');
        }
    });
    
    // Plugin update
    document.querySelector('.update-plugin')?.addEventListener('click', function() {
        const plugin = this.dataset.plugin;
        const btn = this;
        const originalText = btn.innerHTML;
        
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Updating...</span>';
        btn.disabled = true;
        
        fetch(`{{ route('admin.plugins.update', '') }}/${plugin}`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('success', data.message);
                setTimeout(() => location.reload(), 2000);
            } else {
                showAlert('error', data.message);
            }
        })
        .catch(error => {
            showAlert('error', 'An error occurred while updating the plugin.');
        })
        .finally(() => {
            btn.innerHTML = originalText;
            btn.disabled = false;
        });
    });
    
    // Import configuration modal
    const importModal = document.getElementById('importConfigModal');
    const importBtn = document.getElementById('importConfigBtn');
    const confirmImportBtn = document.getElementById('confirmImportBtn');
    const importForm = document.getElementById('importConfigForm');
    
    importBtn?.addEventListener('click', function() {
        importModal.classList.add('active');
    });
    
    confirmImportBtn?.addEventListener('click', function() {
        const formData = new FormData(importForm);
        const fileInput = document.getElementById('configFile');
        
        if (!fileInput.files.length) {
            showAlert('error', 'Please select a configuration file.');
            return;
        }
        
        const btn = this;
        const originalText = btn.textContent;
        btn.textContent = 'Importing...';
        btn.disabled = true;
        
        fetch(`{{ route('admin.plugins.config.import', $pluginData['name']) }}`, {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('success', data.message);
                setTimeout(() => location.reload(), 2000);
            } else {
                showAlert('error', data.message);
            }
        })
        .catch(error => {
            showAlert('error', 'An error occurred while importing configuration.');
        })
        .finally(() => {
            btn.textContent = originalText;
            btn.disabled = false;
            importModal.classList.remove('active');
        });
    });
    
    // Uninstall plugin modal
    const uninstallModal = document.getElementById('uninstallModal');
    const uninstallBtn = document.getElementById('uninstallPluginBtn');
    const confirmUninstallBtn = document.getElementById('confirmUninstallBtn');
    
    uninstallBtn?.addEventListener('click', function() {
        uninstallModal.classList.add('active');
    });
    
    confirmUninstallBtn?.addEventListener('click', function() {
        const btn = this;
        const originalText = btn.textContent;
        btn.textContent = 'Uninstalling...';
        btn.disabled = true;
        
        fetch(`{{ route('admin.plugins.uninstall', $pluginData['name']) }}`, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('success', data.message);
                setTimeout(() => {
                    window.location.href = '{{ route("admin.plugins.index") }}';
                }, 2000);
            } else {
                showAlert('error', data.message);
            }
        })
        .catch(error => {
            showAlert('error', 'An error occurred while uninstalling the plugin.');
        })
        .finally(() => {
            btn.textContent = originalText;
            btn.disabled = false;
            uninstallModal.classList.remove('active');
        });
    });
    
    // Modal close handlers
    document.querySelectorAll('.close-btn, .close-modal').forEach(btn => {
        btn.addEventListener('click', function() {
            importModal.classList.remove('active');
            uninstallModal.classList.remove('active');
        });
    });
    
    // Close modals when clicking backdrop
    [importModal, uninstallModal].forEach(modal => {
        modal?.addEventListener('click', function(e) {
            if (e.target === this) {
                this.classList.remove('active');
            }
        });
    });
    
    // Configuration form submission
    document.getElementById('configForm')?.addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const submitBtn = this.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
        submitBtn.disabled = true;
        
        fetch(this.action, {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('success', data.message || 'Configuration saved successfully!');
            } else {
                showAlert('error', data.message || 'Failed to save configuration.');
            }
        })
        .catch(error => {
            showAlert('error', 'An error occurred while saving configuration.');
        })
        .finally(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    });
    
    // Alert system
    function showAlert(type, message) {
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type === 'success' ? 'success' : type === 'error' ? 'danger' : 'info'} alert-dismissible`;
        alertDiv.innerHTML = `
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
            ${message}
            <button type="button" class="alert-close" onclick="this.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        `;
        
        document.querySelector('.dashboard-header').insertAdjacentElement('afterend', alertDiv);
        
        setTimeout(() => {
            if (alertDiv.parentElement) {
                alertDiv.remove();
            }
        }, 5000);
    }
});
</script>
@endpush